// In this sketch, external images to define the brush are required.
// Please create "assets" forlder under your project folder.
// All images you use as brushes must be in the "assets" folder.

////////////////////////////////////////////////////////////////////////////////
// parameters                                                                 //
////////////////////////////////////////////////////////////////////////////////

// wether or not to show the environment, the vehicles and their traces
var showVehicles = true;
var showDrawing = true;
var showEnvironment = true;
var showSensors = true;

// number of vehicles
var vehicleNum = 10;

////////////////////////////////////////////////////////////////////////////////
// globals                                                                    //
////////////////////////////////////////////////////////////////////////////////

// list of all vehicles
var vehicles;

// pixel density of the display
var density;

// graphics
var canvas, environment, drawing;

// graphics for brush
var brushBlack, brushWhite;

// add a GUI
var gui;

// loading images for brush
// I used 50 * 50 px images and they worked nicely.
function preload() {
  brushBlack = loadImage("assets/Brush_black.png");
  brushWhite = loadImage("assets/Brush_white.png");
}

function setup() {

  // account for retina displays
  density = pixelDensity();

  setupGraphics();
  setupVehicles();

  gui = createGui('braitenberg vehicles 2.a', width - 220, 20);
  gui.addGlobals('showVehicles', 'showDrawing', 'showEnvironment');

  // set default environment (black and white noise)
  environment.background(0);
  environment.strokeWeight(1);
  var noiseScale = 100;
  noiseSeed(random(5));
  environment.loadPixels();
  for (var y = 0; y < environment.height; y++) {
    for (var x = 0; x < environment.width; x++) {
      var noiseColor = map(noise(x / noiseScale, y / noiseScale), 0.3, 0.7, 0, 255);
      environment.set(x, y, noiseColor);
    }
  }
  environment.updatePixels();

}


function setupGraphics() {
  var density = pixelDensity();
  // canvas where our vehicles are drawn
  canvas = createCanvas(windowWidth, windowHeight);
  // environment sensed by the vehicles
  environment = createGraphics(width * density, height * density);
  // drawing created by the vehicles
  drawing = createGraphics(width, height);
}


function setupVehicles() {
  // create some vehicles and add them to the list
  vehicles = [];
  for (var i = 0; i < vehicleNum; i++) {
    // if the last argument is 0, the vehicle prefers white. if it's 1, it prefers black. 
    var v = createVehicle(random(width), random(height), i % 2);
    vehicles.push(v);
  }
}

function draw() {

  // move the vehicles around
  for (var i = 0; i < vehicles.length; i++) {
    vehicles[i].move();
  }

  // now draw stuff
  background(255);
  noStroke();

  // update the environment
  environment.tint(255, 50);

  // show the environment on screen
  if (showEnvironment) {
    image(environment);
  }

  if (showDrawing) {
    image(drawing);
  }
  
  // draw the streaks with brushes
  environment.imageMode(CENTER);
  for (i = 0; i < vehicles.length; i++) {
    if (vehicles[i].getColor() === 0) {
      // if a vehicle prefers white, it draws with white brush
      environment.image(brushWhite, vehicles[i].getX(), vehicles[i].getY(), brushWhite.width, brushWhite.height);
    } else {
      // if a vehicle prefers black, it draws with black brush
      environment.image(brushBlack, vehicles[i].getX(), vehicles[i].getY(), brushBlack.width, brushBlack.height);
    }
  }

  // draw the vehicles on screen
  if (showVehicles) {
    for (i = 0; i < vehicles.length; i++) {
      vehicles[i].show();
    }
  }


}

function createVehicle(x, y, color) {

  // orientation of the vehicle
  var angle = random(360);

  // speed of the vehicle
  var speed = 1;
  var maxSpeed = 5;

  // sensor measurements
  var leftLight = 0;
  var rightLight = 0;

  // sensors
  var leftEye;
  var rightEye;

  // width and height of the vehicle
  var w = 10;
  var h = 20;

  // diameter of those beautiful eyes
  var dEye = 7;

  // previous positions
  var px = 0;
  var py = 0;

  function show() {

    rectMode(CENTER);

    push();

    // move coordinate system to the center of the vehicle
    translate(x, y);

    // draw a rectangle heading forward
    if (color === 0) {
      // if vehicle prefers white...
      stroke(0);
      fill(200, 200, 200, 200);
    } else {
      // if vehicle prefers black...
      stroke(255);
      fill(0, 0, 0, 200);
    }
    rotate(angle + HALF_PI);
    rect(0, 0, w, h);

    pop();

    // draw the eyes
    if (showSensors) {
      var leftEye = sensorPosition(PI / 12);
      var rightEye = sensorPosition(-PI / 12);
      stroke(0);
      strokeWeight(0.5);
      fill(230, 153, 25, leftLight);
      ellipse(leftEye.x, leftEye.y, dEye, dEye);
      fill(230, 153, 25, rightLight);
      ellipse(rightEye.x, rightEye.y, dEye, dEye);
    }

  }

  function move() {
    // use your sensors
    sense();

    // keep track of our previous position
    px = x;
    py = y;

    // move forward
    x += cos(angle) * speed;
    y += sin(angle) * speed;

    // vehicles will switch the direction if they reach at the edge of the screen
    if (x < 30 || x > width - 30 || y < 30 || y > height - 30) {
      angle += 0.3;
    }

    // leave a trace behind
    drawing.stroke(0);
    drawing.strokeWeight(0.25);
    drawing.line(px / density, py / density, x / density, y / density);

  }

  function sensorPosition(direction) {

    var sensorAngle = angle + direction;

    // return positon based on vehicle position
    var distance = max(w, h);

    return {
      x: x + cos(sensorAngle) * distance,
      y: y + sin(sensorAngle) * distance
    };
  }

  function sense() {

    // sensor positions at +/- 15 degrees
    leftEye = sensorPosition(-15);
    rightEye = sensorPosition(+15);

    // sensor measurements (just use the red channel)
    if (color === 0) {
      // if vehicle prefers white...
      leftLight = red(environment.get(leftEye.x / density, leftEye.y / density));
      rightLight = red(environment.get(rightEye.x / density, rightEye.y / density));
    } else {
      // if vehicle prefers black...
      leftLight = map(red(environment.get(leftEye.x / density, leftEye.y / density)), 0, 255, 255, 0);
      rightLight = map(red(environment.get(rightEye.x / density, rightEye.y / density)), 0, 255, 255, 0);
    }

    // speed is directly proportional to the intensity of the input
    leftSpeed = map(rightLight, 0, 255, 0, maxSpeed);
    rightSpeed = map(leftLight, 0, 255, 0, maxSpeed);

    // calculate resulting driving angle
    angle += atan2(leftSpeed - rightSpeed, w);
    speed = (leftSpeed + rightSpeed) / 2;

  }
  
  // to feed x coordinates to the brushes
  function getX() {
    return x;
  }
  
  // to feed y coordinates to the brushes
  function getY() {
    return y;
  }
  
  // to feed color information to the brushes
  function getColor() {
    return color;
  }

  // expose functions by returning an object that encapsulates them.
  // (no need for 'this' or 'new' ...)
  return {
    draw: draw,
    show: show,
    move: move,
    getX: getX,
    getY: getY,
    getColor: getColor
  };

}

function keyTyped() {
  setupVehicles();
  drawing.clear();
}